/*
 * Decompiled with CFR 0.152.
 */
package tools.mv.backend.util;

import com.google.gson.Gson;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.concurrent.locks.ReentrantLock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import tools.mv.backend.drive.BaseDrive;
import tools.mv.backend.drive.Box;
import tools.mv.backend.drive.Dropbox;
import tools.mv.backend.drive.FTP;
import tools.mv.backend.drive.GoogleDrive;
import tools.mv.backend.drive.LocalDrive;
import tools.mv.backend.drive.MediaFire;
import tools.mv.backend.drive.OneDrive;
import tools.mv.backend.drive.PCloud;
import tools.mv.backend.drive.SFTP;
import tools.mv.backend.drive.WebDAV;
import tools.mv.backend.entity.Drive;
import tools.mv.backend.exception.BusinessException;
import tools.mv.backend.exception.DriveLimitException;
import tools.mv.backend.model.AppDrive;
import tools.mv.backend.model.AppInfo;
import tools.mv.backend.model.AuthParam;
import tools.mv.backend.model.DeleteParam;
import tools.mv.backend.model.DownloadParam;
import tools.mv.backend.model.DownloadResult;
import tools.mv.backend.model.DriveAuth;
import tools.mv.backend.model.DriveInfo;
import tools.mv.backend.model.ListParam;
import tools.mv.backend.model.ListResult;
import tools.mv.backend.model.MkdirParam;
import tools.mv.backend.model.Node;
import tools.mv.backend.model.RenameParam;
import tools.mv.backend.model.UploadParam;
import tools.mv.backend.service.AppService;
import tools.mv.backend.service.DriveService;
import tools.mv.backend.task.ListAllError;
import tools.mv.backend.task.TaskProgress;
import tools.mv.backend.util.StringUtil;

public class DriveUtil
implements AutoCloseable {
    private final AppService appService;
    private final DriveService driveService;
    private final String id;
    private final String type;
    private final AppInfo info;
    private DriveAuth auth;
    private final BaseDrive baseDrive;
    private final TaskProgress progress;
    private final ReentrantLock lock = new ReentrantLock();
    private static final Logger logger = LoggerFactory.getLogger(DriveUtil.class);

    public DriveUtil(AppService appService, DriveService driveService, Drive drive, TaskProgress progress) throws Exception {
        this.appService = appService;
        this.driveService = driveService;
        this.id = drive.getId();
        this.type = drive.getType();
        String infoJson = drive.getInfo();
        this.info = (AppInfo)new Gson().fromJson(infoJson, AppInfo.class);
        this.auth = drive.getDriveAuth();
        this.baseDrive = DriveUtil.createBaseDrive(this.type, AppDrive.builder().info(this.info).auth(this.auth).progress(progress).build());
        this.progress = progress;
    }

    public DriveUtil(AppService appService, DriveService driveService, Drive drive) throws Exception {
        this(appService, driveService, drive, null);
    }

    public DriveUtil(AppService appService, DriveService driveService, String type, AppDrive appDrive) throws Exception {
        this.appService = appService;
        this.driveService = driveService;
        this.id = null;
        this.type = type;
        this.info = appDrive.getInfo();
        this.auth = appDrive.getAuth();
        this.baseDrive = DriveUtil.createBaseDrive(type, appDrive);
        this.progress = appDrive.getProgress();
    }

    private static BaseDrive createBaseDrive(String type, AppDrive appDrive) throws Exception {
        switch (type) {
            case "LocalDrive": {
                return new LocalDrive(appDrive);
            }
            case "FTP": {
                return new FTP(appDrive);
            }
            case "SFTP": {
                return new SFTP(appDrive);
            }
            case "WebDAV": {
                return new WebDAV(appDrive);
            }
            case "GoogleDrive": {
                return new GoogleDrive(appDrive);
            }
            case "OneDrive": {
                return new OneDrive(appDrive);
            }
            case "Dropbox": {
                return new Dropbox(appDrive);
            }
            case "PCloud": {
                return new PCloud(appDrive);
            }
            case "Box": {
                return new Box(appDrive);
            }
            case "MediaFire": {
                return new MediaFire(appDrive);
            }
        }
        throw new BusinessException("Drive not implemented", new Object[0]);
    }

    public String getDriveId() {
        return this.id;
    }

    public String getDriveType() {
        return this.type;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void refreshAuth() throws Exception {
        if (this.id == null) {
            this.baseDrive.refresh();
        } else {
            String string = this.id.intern();
            synchronized (string) {
                DriveAuth driveAuth;
                Drive drive = this.driveService.get(this.id);
                long now = System.currentTimeMillis();
                if (now - drive.getMtime() < 60000L && !(driveAuth = drive.getDriveAuth()).equals(this.auth)) {
                    this.auth = driveAuth;
                    this.baseDrive.newAuth(driveAuth);
                    return;
                }
                DriveAuth auth = this.baseDrive.refresh();
                this.updateAuth(auth);
            }
        }
    }

    private void updateAuth(DriveAuth auth) {
        if (auth != null) {
            this.auth = auth;
            if (this.id != null) {
                Drive drive = this.driveService.get(this.id);
                drive.setAuth(new Gson().toJson((Object)auth));
                this.driveService.update(drive);
            }
        }
    }

    public String auth(AuthParam param) throws Exception {
        return this.baseDrive.auth(param);
    }

    public DriveAuth callback(AuthParam param) throws Exception {
        return this.baseDrive.callback(param);
    }

    public DriveInfo info() throws Exception {
        int errorCount = 0;
        while (true) {
            try {
                return this.baseDrive.info();
            }
            catch (Exception e) {
                ++errorCount;
                if (this.baseDrive.expire(e)) {
                    this.refreshAuth();
                    return this.baseDrive.info();
                }
                if (this.baseDrive.isLimited(e)) {
                    logger.error("Error retrieving target space information: " + this.type + "\uff0c" + this.id + "\uff0c" + String.valueOf(e));
                    Thread.sleep(1000L);
                    if (errorCount != 10) continue;
                    throw e;
                }
                throw e;
            }
            break;
        }
    }

    public ListResult list(ListParam param) throws Exception {
        try {
            return this.baseDrive.list(param);
        }
        catch (Exception e) {
            if (this.baseDrive.expire(e)) {
                this.refreshAuth();
                return this.baseDrive.list(param);
            }
            throw e;
        }
    }

    public ListResult listAll(ListParam param, boolean hasSameName, boolean needSort, ListAllError listAllError) throws Exception {
        Node listNode = param.getNode();
        ArrayList<Node> nodes = new ArrayList<Node>(0);
        int sleepSeconds = 1;
        int attempt = 0;
        boolean isLimited = false;
        do {
            ++attempt;
            try {
                ListResult list = this.list(param);
                nodes.addAll(list.getNodes());
                param.setStart(list.getNext());
                isLimited = false;
                attempt = 0;
            }
            catch (Exception e) {
                logger.error(StringUtil.nowDate() + "_" + this.id + " Folder expansion error: " + listNode.getName() + "\uff0c" + e.toString());
                if (this.baseDrive.isLimited(e) || e instanceof DriveLimitException) {
                    isLimited = true;
                    int finalSeconds = Math.min(sleepSeconds *= 2, 3);
                    if (listAllError != null) {
                        listNode.setMsg("attempt:" + attempt + ",sleepSeconds:" + finalSeconds + ":" + StringUtil.getMsg(e));
                        listAllError.callback(listNode.getMsg());
                    }
                    for (int i = 0; i < finalSeconds; ++i) {
                        if (i % 3 == 0) {
                            this.refreshProgress();
                        }
                        Thread.sleep(1000L);
                    }
                    continue;
                }
                if (this.baseDrive.isWarning(e)) {
                    listNode.setWarn(true);
                }
                throw e;
            }
        } while (param.getStart() != null || isLimited);
        if (hasSameName) {
            this.renameDuplicateNodes(nodes);
        }
        Node parent = param.getNode();
        for (Node node : nodes) {
            node.setParent(parent);
        }
        if (needSort) {
            nodes.sort((o1, o2) -> {
                int res = o1.getType().compareTo(o2.getType());
                if (res == 0) {
                    return o1.getName().toLowerCase().compareTo(o2.getName().toLowerCase());
                }
                return res;
            });
        }
        return ListResult.builder().nodes(nodes).build();
    }

    private void renameDuplicateNodes(List<Node> nodes) {
        HashMap<String, Integer> nameCountMap = new HashMap<String, Integer>();
        for (Node node : nodes) {
            String newName;
            String extension;
            String baseName;
            String originalName = node.getName();
            String nameKey = originalName.toLowerCase();
            if (!nameCountMap.containsKey(nameKey)) {
                nameCountMap.put(nameKey, 1);
                continue;
            }
            int count = (Integer)nameCountMap.get(nameKey);
            int dotIndex = originalName.lastIndexOf(46);
            if (dotIndex != -1) {
                baseName = originalName.substring(0, dotIndex);
                extension = originalName.substring(dotIndex);
            } else {
                baseName = originalName;
                extension = "";
            }
            do {
                newName = baseName + "(" + count + ")" + extension;
                ++count;
            } while (nameCountMap.containsKey(newName.toLowerCase()));
            node.setName(newName);
            nameCountMap.put(nameKey, count);
            nameCountMap.put(newName.toLowerCase(), 1);
        }
    }

    public Node mkdir(MkdirParam param) throws Exception {
        try {
            return this.baseDrive.mkdir(param);
        }
        catch (Exception e) {
            if (this.baseDrive.expire(e)) {
                this.refreshAuth();
                return this.baseDrive.mkdir(param);
            }
            throw e;
        }
    }

    public Node rename(RenameParam param) throws Exception {
        try {
            return this.baseDrive.rename(param);
        }
        catch (Exception e) {
            if (this.baseDrive.expire(e)) {
                this.refreshAuth();
                return this.baseDrive.rename(param);
            }
            throw e;
        }
    }

    public void delete(DeleteParam param) throws Exception {
        try {
            this.baseDrive.delete(param);
        }
        catch (Exception e) {
            if (this.baseDrive.expire(e)) {
                this.refreshAuth();
                this.baseDrive.delete(param);
            }
            throw e;
        }
    }

    public DownloadResult download(DownloadParam param) throws Exception {
        this.lock.lock();
        try {
            return this.baseDrive.download(param);
        }
        catch (Exception e) {
            if (this.baseDrive.expire(e)) {
                this.refreshAuth();
                return this.baseDrive.download(param);
            }
            throw e;
        }
    }

    private void setDownloadMsg(Node node, long cur) {
        Long size = node.getSize();
        if (size == null) {
            node.setMsg("Download " + StringUtil.formatBytes(cur));
        } else {
            node.setMsg("Download " + StringUtil.formatBytes(cur) + " / " + StringUtil.formatBytes(size));
        }
        this.refreshProgress();
    }

    private void refreshProgress() {
        if (this.progress != null) {
            this.progress.callback();
        }
    }

    private File createTempFile(InputStream in, Node node) throws Exception {
        File tempFile = File.createTempFile("file_no_size_", ".tmp");
        try (FileOutputStream fos = new FileOutputStream(tempFile);){
            byte[] buffer = new byte[8192];
            int len = -1;
            long cur = 0L;
            while ((len = in.read(buffer)) != -1) {
                fos.write(buffer, 0, len);
                this.setDownloadMsg(node, cur += (long)len);
            }
            fos.flush();
        }
        return tempFile;
    }

    public Node upload(UploadParam param) throws Exception {
        Object driveUtil;
        this.lock.lock();
        DownloadResult result = param.getResult();
        Node fromNode = result.getNode();
        FileInputStream in = null;
        File file = null;
        if (fromNode.getSize() == null && this.baseDrive.needSize()) {
            file = this.createTempFile(result.getIn(), fromNode);
            file.deleteOnExit();
            fromNode.setSize(file.length());
            in = new FileInputStream(file);
            result.setIn(in);
            driveUtil = result.getDriveUtil();
            if (driveUtil != null) {
                ((DriveUtil)driveUtil).finish();
                result.setDriveUtil(null);
            }
        }
        try {
            driveUtil = this.uploadAndDealAuth(param);
            return driveUtil;
        }
        catch (Exception e) {
            if (this.baseDrive.expire(e)) {
                this.refreshAuth();
                if (in != null) {
                    in.close();
                    in = new FileInputStream(file);
                    result.setIn(in);
                }
                Node node = this.uploadAndDealAuth(param);
                return node;
            }
            throw e;
        }
        finally {
            if (in != null) {
                in.close();
            }
            if (file != null && file.exists()) {
                Files.delete(file.toPath());
            }
        }
    }

    private Node uploadAndDealAuth(UploadParam param) throws Exception {
        Node upload = this.baseDrive.upload(param);
        if (this.baseDrive.refreshed()) {
            DriveAuth auth = this.baseDrive.loadAuth();
            this.updateAuth(auth);
        }
        return upload;
    }

    public String supportName(String name) {
        return this.baseDrive.supportName(name);
    }

    public void finish() throws Exception {
        try {
            this.baseDrive.finish();
        }
        finally {
            if (this.lock.isHeldByCurrentThread()) {
                this.lock.unlock();
            }
        }
    }

    @Override
    public void close() {
        this.baseDrive.close();
    }

    public boolean isLimited(Exception e) {
        return this.baseDrive.isLimited(e) || e instanceof DriveLimitException;
    }

    public boolean isWarning(Exception e) {
        return this.baseDrive.isWarning(e);
    }
}

